from flask import Flask, render_template, request, redirect, url_for, flash
from flask_login import LoginManager, login_user, login_required, logout_user, current_user
from database import db
from models import User, TestAttempt
import json


app = Flask(__name__)
app.config['SECRET_KEY'] = 'your-secret-key'
app.config['SQLALCHEMY_DATABASE_URI'] = 'sqlite:///app.db'
db.init_app(app)

# Create tables
with app.app_context():
    db.create_all()

with open('data/questions.json') as f:
    questions = json.load(f)

# Flask-Login setup
login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'login'

@login_manager.user_loader
def load_user(user_id):
    return User.query.get(int(user_id))

# Routes for login/register
@app.route('/login', methods=['GET', 'POST'])
def login():
    if request.method == 'POST':
        username = request.form['username']
        password = request.form['password']
        user = User.query.filter_by(username=username).first()
        if user and user.check_password(password):
            login_user(user)
            return redirect(url_for('dashboard'))
        flash('Invalid credentials')
    return render_template('login.html')

@app.route('/register', methods=['GET', 'POST'])
def register():
    if request.method == 'POST':
        username = request.form['username']
        password = request.form['password']
        if User.query.filter_by(username=username).first():
            flash('Username already exists')
            return redirect(url_for('register'))
        user = User(username=username)
        user.set_password(password)
        db.session.add(user)
        db.session.commit()
        flash('Registration successful! Please log in.')
        return redirect(url_for('login'))
    return render_template('register.html')

@app.route('/logout')
@login_required
def logout():
    logout_user()
    return redirect(url_for('login'))

# Protected route (only for logged-in users)
@app.route('/dashboard')
@login_required
def dashboard():
    # Get available tests (excluding those already taken)
    user_attempts = {a.test_name for a in current_user.attempts}
    available = [t for t in questions.keys() if t not in user_attempts]
    
    # Get test history
    attempts = TestAttempt.query.filter_by(
        user_id=current_user.id
    ).order_by(
        TestAttempt.timestamp.desc()
    ).all()
    
    return render_template('dashboard.html',
                        available_tests=available,
                        current_user=current_user,
                        attempts=attempts,
                        questions=questions)

# Update quiz route to require 

@app.route('/')
def welcome():
    return render_template('welcome.html')

@app.route('/quiz/<test_name>', methods=['GET', 'POST'])
@login_required
def quiz(test_name):
    # Get the specific test data
    existing = TestAttempt.query.filter_by(
    user_id=current_user.id,
    test_name=test_name
    ).first()
    
    if existing:
        flash("You've already taken this test!")
        return redirect(url_for('dashboard'))
    
    test_data = questions.get(test_name, {})

   
    if request.method == 'POST':
        score = 0
        for q_id, q_data in test_data.items():
            if isinstance(q_data, dict):  # Only process question dictionaries
                user_answer = request.form.get(q_id)
                if user_answer == q_data.get('correct_answer'):
                    score += 1
        attempt = TestAttempt(
        user_id=current_user.id,
        test_name=test_name,
        score=score
            )
        db.session.add(attempt)
        db.session.commit()
    

        return redirect(url_for('result', test_name=test_name, score=score))
    
    # Filter only question dictionaries for the GET request
    quiz_questions = {k:v for k,v in test_data.items() if isinstance(v, dict)}
    return render_template('quiz.html', 
                         test_name=test_name,
                         questions=quiz_questions)

@app.route('/result/<test_name>/<int:score>')
@login_required
def result(test_name, score):
    total = len(questions[test_name])
    return render_template('result.html', score=score, total=total, test_name=test_name)

@app.route('/theory/<test_name>')
@login_required
def theory(test_name):
    return render_template('theory.html', theory_content=questions[test_name]["theory"])

@app.route('/admin')
@login_required
def admin():
    if not current_user.is_admin:  # Add this field to your User model
        abort(403)
    
    attempts = TestAttempt.query.order_by(TestAttempt.timestamp.desc()).all()
    return render_template('admin.html', attempts=attempts)

@app.cli.command('create-admin')
def create_admin_command():
    """Create the admin user"""
    from models import User
    admin = User.query.filter_by(username='admin').first()
    if not admin:
        admin = User(
            username='admin',
            is_admin=True
        )
        admin.set_password(input("Enter admin password: "))
        db.session.add(admin)
        db.session.commit()
        print("Admin user created!")
    else:
        print("Admin user already exists")

if __name__ == '__main__':
    app.run(debug=True)